/*
 * Decompiled with CFR 0.152.
 */
package silly511.backups.helpers;

import com.google.common.collect.ImmutableList;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.function.Supplier;
import javax.imageio.ImageIO;
import net.minecraft.nbt.CompressedStreamTools;
import net.minecraft.nbt.NBTTagCompound;
import silly511.backups.BackupsMod;
import silly511.backups.Config;
import silly511.backups.helpers.FileHelper;
import silly511.backups.util.GzipInputStream;
import silly511.backups.util.GzipOutputStream;
import silly511.backups.util.IORunnable;

public final class BackupHelper {
    public static final DateTimeFormatter dateFormat = DateTimeFormatter.ofPattern("yyyy-M-d k-mm-ss");
    public static final int FORMAT_VERSION = 1;

    public static Backup backup(Path sourceDir, Path backupsDir, BackupReason reason, Supplier<BufferedImage> iconFetcher) throws IOException {
        Path currentBackup = backupsDir.resolve("In-Progress");
        Path lastBackup = BackupHelper.getLastBackup(backupsDir);
        Instant time = Instant.now();
        Files.createDirectories(currentBackup, new FileAttribute[0]);
        FileHelper.cleanDirectory(currentBackup);
        for (Path sourceFile : FileHelper.listFilesDeep(sourceDir, false)) {
            Path currentFile = FileHelper.relativize(sourceDir, sourceFile, currentBackup);
            if (Files.isDirectory(sourceFile, LinkOption.NOFOLLOW_LINKS)) {
                Files.createDirectory(currentFile, new FileAttribute[0]);
                continue;
            }
            if (!Files.isRegularFile(sourceFile, LinkOption.NOFOLLOW_LINKS)) continue;
            Path lastFile = FileHelper.relativizeAdd(sourceDir, sourceFile, lastBackup, ".gz");
            currentFile = currentFile.resolveSibling(currentFile.getFileName() + ".gz");
            if (Files.isRegularFile(lastFile, LinkOption.NOFOLLOW_LINKS) && Files.getLastModifiedTime(sourceFile, new LinkOption[0]).equals(FileHelper.readGzipTime(lastFile))) {
                Files.createLink(currentFile, lastFile);
                continue;
            }
            try (GzipOutputStream out = new GzipOutputStream(Files.newOutputStream(currentFile, new OpenOption[0]), Files.getLastModifiedTime(sourceFile, LinkOption.NOFOLLOW_LINKS), 8192);){
                Files.copy(sourceFile, out);
            }
        }
        Path finalBackupDir = backupsDir.resolve(time.atZone(ZoneId.systemDefault()).format(dateFormat));
        Files.move(currentBackup, finalBackupDir, new CopyOption[0]);
        Backup backup = new Backup(finalBackupDir.toAbsolutePath().normalize(), 1, reason, time, "1.12.2", null);
        backup.writeBackup();
        if (iconFetcher != null) {
            ImageIO.write((RenderedImage)iconFetcher.get(), "png", finalBackupDir.resolve("icon.png").toFile());
        }
        BackupHelper.setLastBackup(backupsDir, finalBackupDir);
        return backup;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void restoreBackup(Path backupDir, Path targetDir, Path tempDir, Predicate<String> filter) throws IOException {
        Files.createDirectories(tempDir, new FileAttribute[0]);
        Path tempRestoreDir = Files.createTempDirectory(tempDir, "backupRestore", new FileAttribute[0]);
        try {
            LinkedList<IORunnable> attributeCopyTasks = new LinkedList<IORunnable>();
            for (Path backupFile : FileHelper.listFilesDeep(backupDir, false)) {
                Path targetFile = FileHelper.relativizeRemove(backupDir, backupFile, tempRestoreDir, ".gz");
                if (Files.isDirectory(backupFile, new LinkOption[0])) {
                    Files.createDirectory(targetFile, new FileAttribute[0]);
                    continue;
                }
                if (!Files.isRegularFile(backupFile, LinkOption.NOFOLLOW_LINKS) || !backupFile.toString().endsWith(".gz") || filter != null && filter.test(targetFile.getFileName().toString())) continue;
                try (GzipInputStream stream = new GzipInputStream(Files.newInputStream(backupFile, new OpenOption[0]), 8192);){
                    Files.copy(stream, targetFile, StandardCopyOption.REPLACE_EXISTING);
                    attributeCopyTasks.add(() -> Files.setLastModifiedTime(targetFile, stream.getModTime()));
                }
            }
            for (IORunnable t : attributeCopyTasks) {
                t.run();
            }
            Path oldTargetDir = tempDir.resolve("BackupsModOldWorldTemp");
            FileHelper.deleteIfExists(oldTargetDir);
            if (Files.exists(targetDir, LinkOption.NOFOLLOW_LINKS)) {
                Files.move(targetDir, oldTargetDir, new CopyOption[0]);
            }
            Files.createDirectories(targetDir.getParent(), new FileAttribute[0]);
            Files.move(tempRestoreDir, targetDir, new CopyOption[0]);
            FileHelper.deleteIfExists(oldTargetDir);
        }
        finally {
            FileHelper.deleteIfExists(tempRestoreDir);
        }
    }

    public static void trimBackups(Path backupsDir) throws IOException {
        HashSet<Long> set1 = new HashSet<Long>();
        HashSet<Long> set2 = new HashSet<Long>();
        HashSet<Long> set3 = new HashSet<Long>();
        ZoneId timeZone = ZoneId.systemDefault();
        long currentSec = Instant.now().getEpochSecond();
        int i = 0;
        for (Backup backup : BackupHelper.listAllBackups(backupsDir)) {
            if (backup.label != null) continue;
            long day = backup.time.atZone(timeZone).toLocalDate().toEpochDay();
            long sec = backup.time.getEpochSecond();
            long secAgo = currentSec - sec;
            boolean shouldDelete = false;
            if (Config.trimming.maxNumber > 0 && ++i > Config.trimming.maxNumber) {
                shouldDelete = true;
            } else if (Config.trimming.maxAge > 0 && secAgo >= (long)(Config.trimming.maxAge * 86400)) {
                shouldDelete = true;
            } else if (secAgo >= (long)(Config.trimming.perWeek * 86400)) {
                shouldDelete = !set1.add((day + 3L) / 7L);
            } else if (secAgo >= (long)(Config.trimming.perDay * 86400)) {
                shouldDelete = !set2.add(day);
            } else if (secAgo >= (long)(Config.trimming.perHour * 3600)) {
                boolean bl = shouldDelete = !set3.add(sec / 3600L);
            }
            if (!shouldDelete) continue;
            try {
                BackupHelper.deleteBackup(backup);
                BackupsMod.logger.info("Trimming backup " + backup.time);
            }
            catch (IOException ex) {
                BackupsMod.logger.error("Unable to trim backup " + backup.time, (Throwable)ex);
            }
        }
    }

    public static void deleteBackup(Backup backup) throws IOException {
        Path backupsDir = backup.dir.getParent();
        if (BackupHelper.getLastBackup(backupsDir).equals(backup.dir)) {
            List<Backup> backups = BackupHelper.listAllBackups(backupsDir);
            int index = backups.indexOf(backup);
            if (backups.size() > 1) {
                BackupHelper.setLastBackup(backupsDir, backups.get((int)(index + (index == 0 ? 1 : -1))).dir);
            }
        }
        FileHelper.deleteDirectory(backup.dir);
    }

    public static void setLastBackup(Path backupsDir, Path newLastBackup) throws IOException {
        Path last = backupsDir.resolve("Last");
        if (Files.isSymbolicLink(last)) {
            Files.delete(last);
        }
        Files.write(last, newLastBackup.getFileName().toString().getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
    }

    public static Path getLastBackup(Path backupsDir) throws IOException {
        Path readDir;
        Path last = backupsDir.resolve("Last");
        if (Files.isRegularFile(last, new LinkOption[0]) && Files.isDirectory(readDir = backupsDir.resolve(new String(Files.readAllBytes(last), StandardCharsets.UTF_8)), new LinkOption[0])) {
            return readDir;
        }
        return last;
    }

    public static List<Backup> listAllBackups(Path backupsDir) {
        if (!Files.isDirectory(backupsDir, new LinkOption[0])) {
            return ImmutableList.of();
        }
        ArrayList<Backup> list = new ArrayList<Backup>();
        try {
            for (Path path : FileHelper.listFiles(backupsDir)) {
                if (!Files.isRegularFile(path.resolve("backupMetadata.dat"), new LinkOption[0])) continue;
                list.add(Backup.getBackup(path));
            }
        }
        catch (IOException ex) {
            BackupsMod.logger.error("Error trying to list backups", (Throwable)ex);
        }
        list.sort((b1, b2) -> b2.time.compareTo(b1.time));
        return ImmutableList.copyOf(list);
    }

    public static class Backup {
        public final Path dir;
        public final int format;
        public final BackupReason reason;
        public final Instant time;
        public final String mcVersion;
        protected String label;
        private static final Map<Path, Backup> cache = new HashMap<Path, Backup>();

        public static Backup getBackup(Path backupDir) {
            Path metadataFile = (backupDir = backupDir.toAbsolutePath().normalize()).resolve("backupMetadata.dat");
            if (!Files.isRegularFile(metadataFile, new LinkOption[0])) {
                return null;
            }
            if (cache.containsKey(backupDir)) {
                return cache.get(backupDir);
            }
            try {
                NBTTagCompound tag = CompressedStreamTools.func_74797_a((File)metadataFile.toFile());
                BackupReason reason = BackupReason.values()[tag.func_74771_c("Reason")];
                Instant time = Instant.ofEpochSecond(tag.func_74763_f("Time"));
                String label = tag.func_150297_b("Label", 8) ? tag.func_74779_i("Label") : null;
                return new Backup(backupDir, tag.func_74762_e("Format"), reason, time, tag.func_74779_i("mcVersion"), label);
            }
            catch (IOException | IndexOutOfBoundsException ex) {
                return new Backup(backupDir, 1, null, FileHelper.getDateCreated(backupDir), null, null);
            }
        }

        protected Backup(Path dir, int format, BackupReason reason, Instant time, String mcVersion, String label) {
            this.dir = dir;
            this.format = format;
            this.reason = reason;
            this.time = time;
            this.mcVersion = mcVersion;
            this.label = label;
            cache.put(this.dir, this);
        }

        public void writeBackup() throws IOException {
            NBTTagCompound tag = new NBTTagCompound();
            tag.func_74768_a("Format", this.format);
            tag.func_74774_a("Reason", (byte)this.reason.ordinal());
            tag.func_74772_a("Time", this.time.getEpochSecond());
            tag.func_74778_a("mcVersion", this.mcVersion);
            if (this.label != null) {
                tag.func_74778_a("Label", this.label);
            }
            CompressedStreamTools.func_74795_b((NBTTagCompound)tag, (File)this.dir.resolve("backupMetadata.dat").toFile());
        }

        public void setLabel(String label) {
            this.label = label;
        }

        public String getLabel() {
            return this.label;
        }
    }

    public static enum BackupReason {
        SCHEDULED,
        USER,
        RESTORE,
        WORLD_JOIN;

        public final String tranKey = "backups.reason." + this.name().toLowerCase();
    }
}

